/* ------------------------------------------------------------------------ */
/*  @@ Source Documentation                         *** C Version ***       */
/*                                                                          */
/*  TITLE : DEMOMIX.C                                                       */
/*                                                                          */
/*  DESCRIPTION :                                                           */
/*      This program demostrates how to use the AUXDRV.DRV driver to        */
/*      get and set the mixer setting.                                      */
/*      file.                                                               */
/*                                                                          */
/*      You need to have Port Blaster, Sound Blaster Pro card or higher     */
/*      to run this program.                                                */
/*                                                                          */
/*      Note that the BLASTER environment has to be set before executing    */
/*      this program.                                                       */
/*                                                                          */
/*  Copyright (c) Creative Technology Ltd, 1993. All rights reserved.       */
/*                                                                          */
/* ------------------------------------------------------------------------ */

#include <dos.h>
#include <bios.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <conio.h>

#include "ctstddef.h"
#include "sbkaux.h"
#include "sbkmacro.h"
#include "sbkx.h"

#include "loaddrv.c"

/* Mixer setting structure */
struct  structMIXER {
    DWORD   inpSwitch ;      /* Input source switches */
    DWORD   outSwitch ;      /* Output source switches */
    WORD    stAGC ;          /* AGC status - on oR off */
    WORD    inpGain ;        /* Input gain - Range (0 - 3) */
    WORD    outGain ;        /* Output gain - Range (0 - 3) */
    WORD    Bass ;           /* Bass level - Range (0 - 255) */
    WORD    Treble ;         /* Treble level - Range (0 - 255) */
    WORD    volMaster ;      /* Master volume level - Range (0 - 255) */
    WORD    volVoice ;       /* Voice volume level - Range (0 -255) */
    WORD    volMidi ;        /* Midi volume level - Range (0 -255) */
    WORD    volCD ;          /* CD volume level - Range (0 -255) */
    WORD    volLine ;        /* Line in volume level - Range (0 -255) */
    WORD    volMic ;         /* Mic volume level - Range (0 -255) */
    WORD    volPcSpker ;     /* PC speaker volume level - Range (0 -255) */
};


/* Local functions */
int PrepareCTAUXDrv(char * BlasterEnv) ;
void GetMixer(struct structMIXER *ptrMixer) ;
void SetMixer(struct structMIXER *ptrMixer) ;
void PrepareMixerSetting(struct structMIXER *ptrMixer) ;
void DisplayMixer(struct structMIXER *ptrMixer) ;


char far * lpOrigin ; /* Origin pointer used to free driver buffer */

void main ()
{
    struct structMIXER prevSetting ;
    struct structMIXER currSetting ;
    char * BlasterEnv ;


    /* Retrieve the BLASTER environment settings */
    if ((BlasterEnv = getenv("BLASTER")) != NULL)
    {
        /* Loads AUXDRV.DRV into memory */
        if (!PrepareCTAUXDrv(BlasterEnv))
        {
            /* Initialises AUXDRV.DRV */
            ctadInit() ;

            /* Stores mixer setting */
            printf("\n\n\nSaving current mixer settings...") ;
            GetMixer(&prevSetting) ;

            /* Displays mixer setting */
            printf("\nPress a key to display current mixer settings...") ;
            getch() ;
            DisplayMixer(&prevSetting) ;

            /* Fills in the mixer structure */
            PrepareMixerSetting(&currSetting) ;

            /* Sets mixer */
            printf("\nChanging the mixer settings...") ;
            SetMixer(&currSetting) ;
            printf("\nPress a key to view the changes on mixer settings...") ;
            getch() ;

            /* Displays mixer setting */
            DisplayMixer(&currSetting) ;

            /* Restores previous mixer setting */
            printf("\nRestoring back the previous mixer settings...");
            SetMixer(&prevSetting) ;
            printf("\nPress a key to display current mixer settings...") ;
            getch() ;

            /* Display mixer settings */
            DisplayMixer(&prevSetting) ;

            /* Terminates AUXDRV.DRV */
            ctadTerminate() ;

            printf ("\nEnd...") ;
        }
    }
    else
        printf("BLASTER environment not set.\n");
}


/****************************************************************************/
/*  @@ Usage                                                                */
/*                                                                          */
/*  PrepareCTAUXDrv(char * BlasterEnv)                                      */
/*                                                                          */
/*  Description :                                                           */
/*      Load and endorse AUXDRV.DRV.                                        */
/*                                                                          */
/*  Entry :                                                                 */
/*      none                                                                */
/*                                                                          */
/*  Exit :                                                                  */
/*      zero if sucessful, non-zero otherwise.                              */
/*                                                                          */
/****************************************************************************/

int PrepareCTAUXDrv(char * BlasterEnv)
{
    if ((CTAuxDrv = sbkLoadDriver("AUXDRV.DRV",UNUSED,
			(char far **)&lpOrigin)) != NULL)
    {
        /* Retrieves AUXDRV.DRV version */
        if (ctadGetDrvVer() >= 0x0303)
        {
            /* Passes BLASTER environment settings to the driver */
            if (!ctadGetEnvSettings((char far *)BlasterEnv))
                return(0) ;
            else
                printf("\nBLASTER environment is not valid");
        }
        else
        {
            printf("\nInvalid AUXDRV.DRV - ") ;
            printf("I need AUXDRV.DRV version 3.03 or higher.") ;
        }
    }
    return(1) ;
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*  GetMixer(struct *ptrMixer)                                              */
/*                                                                          */
/*  Description :                                                           */
/*      Retrieves and stores the mixer setting.                             */
/*                                                                          */
/*  Entry :                                                                 */
/*      ptrMixer - pointer of the mixer structure where the mixer           */
/*                 setting is to be stored.                                 */
/*                                                                          */
/*  Exit :                                                                  */
/*      none.                                                               */
/* ------------------------------------------------------------------------ */

void GetMixer(struct structMIXER *ptrMixer)
{
    /* Gets input sources switch */
    ptrMixer->inpSwitch = ctadGetMixerSwitch(0) ;
    /* Gets output sources switch */
    ptrMixer->outSwitch = ctadGetMixerSwitch(1) ;

    /* Gets AGC status */
    ptrMixer->stAGC = ctadGetAGC() ;

    /* Get mixer input gain */
    ptrMixer->inpGain = ctadGetMixerGain(0) ;
    /* Get mixer output gain */
    ptrMixer->outGain = ctadGetMixerGain(1) ;

    /* Gets mixer bass level */
    ptrMixer->Bass = ctadGetToneLevel(1) ;
    /* Gets mixer treble level */
    ptrMixer->Treble = ctadGetToneLevel(0) ;

    /* Gets master, voice, CD, line-in, mic and PC speaker volume level */
    ptrMixer->volMaster = ctadGetVolume(MIXERVOL_MASTER) ;
    ptrMixer->volVoice = ctadGetVolume(MIXERVOL_VOICE) ;
    ptrMixer->volMidi = ctadGetVolume(MIXERVOL_MIDI) ;
    ptrMixer->volCD = ctadGetVolume(MIXERVOL_CD) ;
    ptrMixer->volLine = ctadGetVolume(MIXERVOL_LINE) ;
    ptrMixer->volMic = ctadGetVolume(MIXERVOL_MIC) ;
    ptrMixer->volPcSpker = ctadGetVolume(MIXERVOL_PCSPEAKER) ;
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*  SetMixer(struct *ptrMixer)                                              */
/*                                                                          */
/*  Description :                                                           */
/*      Sets mixer.                                                         */
/*                                                                          */
/*  Entry :                                                                 */
/*      ptrMixer - pointer of the mixer structure which specifies           */
/*                 the mixer setting.                                       */
/*                                                                          */
/*  Exit :                                                                  */
/*      none.                                                               */
/* ------------------------------------------------------------------------ */

void SetMixer(struct structMIXER *ptrMixer)
{
    /* Sets input sources switch */
    ctadSetMixerSwitch(0,ptrMixer->inpSwitch) ;
    /* Sets output sources switch */
    ctadSetMixerSwitch(1,ptrMixer->outSwitch) ;

    /* Sets AGC status */
    ctadSetAGC(ptrMixer->stAGC) ;

    /* Set mixer input gain */
    ctadSetMixerGain(0,ptrMixer->inpGain) ;
    /* Set mixer output gain */
    ctadSetMixerGain(1,ptrMixer->outGain) ;

    /* Sets mixer bass level */
    ctadSetToneLevel(1,ptrMixer->Bass) ;
    /* Sets mixer treble level */
    ctadSetToneLevel(0,ptrMixer->Treble) ;

    /* Sets master, voice, CD, line-in, mic and PC speaker volume level */
    ctadSetVolume(MIXERVOL_MASTER,ptrMixer->volMaster) ;
    ctadSetVolume(MIXERVOL_VOICE,ptrMixer->volVoice) ;
    ctadSetVolume(MIXERVOL_MIDI,ptrMixer->volMidi) ;
    ctadSetVolume(MIXERVOL_CD,ptrMixer->volCD) ;
    ctadSetVolume(MIXERVOL_LINE,ptrMixer->volLine) ;
    ctadSetVolume(MIXERVOL_MIC,ptrMixer->volMic) ;
    ctadSetVolume(MIXERVOL_PCSPEAKER,ptrMixer->volPcSpker) ;
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*  PrepareMixerSetting(struct structMIXER *ptrMixer)                       */
/*                                                                          */
/*  Description :                                                           */
/*      Fills in the mixer structure correspondingly.                       */
/*                                                                          */
/*  Entry :                                                                 */
/*      ptrMixer - pointer of the mixer structure.                          */
/*                                                                          */
/*  Exit :                                                                  */
/*      none.                                                               */
/*                                                                          */
/*                                                                          */
/* ------------------------------------------------------------------------ */

void PrepareMixerSetting(struct structMIXER *ptrMixer)
{
    DWORD   dwValue ;

    /* initialises to zero */
    memset(ptrMixer,0,sizeof(struct structMIXER)) ;

    /* sets MIDI as stereo input source */
    dwValue = MIXERSWI_MIDI_R ;
    dwValue |= MIXERSWI_MIDI_L ;
    ptrMixer->inpSwitch = dwValue ;
    ptrMixer->inpSwitch = (ptrMixer->inpSwitch << 16) | dwValue ;

    /* sets both the left and right input gain to 2 */
    ptrMixer->inpGain = (WORD)2 ;
    ptrMixer->inpGain = (WORD)((ptrMixer->inpGain << 8) | 2) ;

    /* sets CD as ouput source */
    ptrMixer->outSwitch = MIXERSWI_CD_R ;
    ptrMixer->outSwitch |= MIXERSWI_CD_L ;

    /* sets both the left and right output gain to 2 */
    ptrMixer->outGain = (WORD)2 ;
    ptrMixer->outGain = (WORD)((ptrMixer->outGain << 8) | 2) ;

    /* sets both the right and left treble level to maximum */
    ptrMixer->Treble = 0xffff ;
    /* sets both the right and left bass level to maximum */
    ptrMixer->Bass = 0xffff ;

    /* sets both the right and left Master volume to 3/4 */
    /* of the maximum level */
    ptrMixer->volMaster = 0xff * 3 / 4 ;
    ptrMixer->volMaster <<= 8 ;
    ptrMixer->volMaster |= (0xff * 3 / 4) ;

    /* sets both the right and left CD volume to 3/4 */
    /* of the maximum level */
    ptrMixer->volCD = 0xff * 3 / 4 ;
    ptrMixer->volCD <<= 8 ;
    ptrMixer->volCD |= (0xff * 3 / 4) ;
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*  DisplayMixer(struct structMIXER *ptrMixer)                              */
/*                                                                          */
/*  Description :                                                           */
/*      Displays mixer setting.                                             */
/*                                                                          */
/*  Entry :                                                                 */
/*      ptrMixer - pointer of the mixer structure which the mixer           */
/*                 setting is to be displayed.                              */
/*                                                                          */
/*  Exit :                                                                  */
/*      none.                                                               */
/* ------------------------------------------------------------------------ */

void DisplayMixer(struct structMIXER *ptrMixer)
{
    printf("\n\t-------------- Mixer Settings ---------------\n\n") ;

    printf("  Input sources\t\tLeft Channel\t\tRight Channel\n");

    printf("\tMic         :\t%s\t\t\t%s\n",
                    ((ptrMixer->inpSwitch >> 16) & MIXERSWI_MIC) ?
                                     "ON" : "OFF",
                    (ptrMixer->inpSwitch & MIXERSWI_MIC) ?
                                     "ON" : "OFF") ;

    printf("\tCD          :\tL:%s\tR:%s\t\tL:%s\tR:%s\n",
                    ((ptrMixer->inpSwitch >> 16) & MIXERSWI_CD_L) ?
                                    "ON" : "OFF",
                    ((ptrMixer->inpSwitch >> 16) & MIXERSWI_CD_R) ?
                                    "ON" : "OFF",
                    (ptrMixer->inpSwitch & MIXERSWI_CD_L) ?
                                    "ON" : "OFF",
                    (ptrMixer->inpSwitch & MIXERSWI_CD_R) ?
                                    "ON" : "OFF") ;

    printf("\tLine        :\tL:%s\tR:%s\t\tL:%s\tR:%s\n",
                    ((ptrMixer->inpSwitch >> 16) & MIXERSWI_LINE_L) ?
                                    "ON" : "OFF",
                    ((ptrMixer->inpSwitch >> 16) & MIXERSWI_LINE_R) ?
                                    "ON" : "OFF",
                    (ptrMixer->inpSwitch & MIXERSWI_LINE_L) ?
                                    "ON" : "OFF",
                    (ptrMixer->inpSwitch & MIXERSWI_LINE_R) ?
                                    "ON" : "OFF") ;

    printf("\tMIDI        :\tL:%s\tR:%s\t\tL:%s\tR:%s\n",
                    ((ptrMixer->inpSwitch >> 16) & MIXERSWI_MIDI_L) ?
                                    "ON" : "OFF",
                    ((ptrMixer->inpSwitch >> 16) & MIXERSWI_MIDI_R) ?
                                    "ON" : "OFF",
                    (ptrMixer->inpSwitch & MIXERSWI_MIDI_L) ?
                                    "ON" : "OFF",
                    (ptrMixer->inpSwitch & MIXERSWI_MIDI_R) ?
                                    "ON" : "OFF") ;

    printf("  Output sources\n");

    printf("\tMic         :\t%s\n",
                    (ptrMixer->outSwitch & MIXERSWI_MIC) ?
                                     "ON" : "OFF") ;

    printf("\tCD          :\tL:%s\tR:%s\n",
                    (ptrMixer->outSwitch & MIXERSWI_CD_L) ?
                                    "ON" : "OFF",
                    (ptrMixer->outSwitch & MIXERSWI_CD_R) ?
                                    "ON" : "OFF") ;

    printf("\tLine        :\tL:%s\tR:%s\n",
                    (ptrMixer->outSwitch & MIXERSWI_LINE_L) ?
                                    "ON" : "OFF",
                    (ptrMixer->outSwitch & MIXERSWI_LINE_R) ?
                                    "ON" : "OFF") ;

    printf("  AGC status        :\t%s\n",
                    ptrMixer->stAGC ? "ON" : "OFF") ;

    printf("  Input gain        :\tL:%d\tR:%d\n",
                    (ptrMixer->inpGain >> 8),
                    (ptrMixer->inpGain & 0x00ff));

    printf("  Output gain       :\tL:%d\tR:%d\n",
                    (ptrMixer->outGain >> 8),
                    (ptrMixer->outGain & 0x00ff));

    printf("  Bass level        :\tL:%d\tR:%d\n",
                    (ptrMixer->Bass >> 8),
                    (ptrMixer->Bass & 0x00ff));

    printf("  Treble level      :\tL:%d\tR:%d\n",
                    (ptrMixer->Treble >> 8),
                    (ptrMixer->Treble & 0x00ff));

    printf("  Master volume     :\tL:%d\tR:%d\n",
                    (ptrMixer->volMaster >> 8),
                    (ptrMixer->volMaster & 0x00ff));

    printf("  Voice volume      :\tL:%d\tR:%d\n",
                    (ptrMixer->volVoice >> 8),
                    (ptrMixer->volVoice & 0x00ff));

    printf("  Midi volume       :\tL:%d\tR:%d\n",
                    (ptrMixer->volMidi >> 8),
                    (ptrMixer->volMidi & 0x00ff));

    printf("  CD volume         :\tL:%d\tR:%d\n",
                    (ptrMixer->volCD >> 8),
                    (ptrMixer->volCD & 0x00ff));

    printf("  Line-in volume    :\tL:%d\tR:%d\n",
                    (ptrMixer->volLine >> 8),
                    (ptrMixer->volLine & 0x00ff));

    printf("  Mic volume        :\t%d\n",ptrMixer->volMic) ;

    printf("  PC speaker volume :\t%d\n",ptrMixer->volPcSpker) ;

}
/* End of file */
